﻿using System;
using System.ComponentModel;
using System.Numerics;

using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.Dep1155Contract")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA Dep1155 Standart NFT implementation")]
    [ContractPermission("*", "*")]
    public class Dep1155Contract : Dep1155Token<TokenState, TokenRoyaltyState>
    {
        // OwnershipTransferred(oldOwner, newOwner)
        public static event Action<UInt160, UInt160> OwnershipTransferred;
        public static event Action<UInt160, Boolean> IsUserBlocked;
        public static event Action<Boolean> IsContractPaused;

        [Safe]
        public override string Symbol() => "NFTdVITA";

        private static readonly ByteString trueInByteString = (ByteString)new byte[] { 0x01 };
        private static readonly byte[] blockListPrefix = new byte[] { 0x01, 0x02 };
        private static readonly byte[] ownerKey = "owner".ToByteArray();
        private static readonly byte[] pausedKey = "paused".ToByteArray();
        private static readonly byte[] marketplaceKey = "marketplace".ToByteArray();
        private static readonly byte[] termsKey = "terms".ToByteArray();
        private static bool IsOwner() => Runtime.CheckWitness(GetOwner());

        public static readonly StorageMap BlocklistMap = new StorageMap(Storage.CurrentContext, blockListPrefix);

        [Safe]
        public static string TermsAndConditions()
        {
            return (string)ContractMap.Get(termsKey);
        }

        [Safe]
        public override Map<string, object> Properties(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            var token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            Map<string, object> map = new();
            map["description"] = token.Description;
            map["name"] = token.Name;
            map["image"] = token.Image;
            map["creator"] = token.Creator;
            map["createdTime"] = token.CreatedTime;
            
            return map;
        }

        public virtual ByteString CreateToken(UInt160 to, BigInteger amount, string data) 
        {
            var tx = (Transaction)Runtime.ScriptContainer;
            Map<string, string> customData = (Map<string, string>)StdLib.JsonDeserialize(data);
            ByteString tokenId = NewTokenId();
            
            Mint(to, tokenId, amount, new TokenState
            {
                Name = customData["name"],
                Creator = tx.Sender,
                Image = customData["image"],
                Description = customData["description"],
                CreatedTime = Runtime.Time
            });

            TokenRoyaltyState tokenRoyalty = new TokenRoyaltyState {Artist = tx.Sender, Percentage = 100};
            SetTokenRoyalty(tokenId, tokenRoyalty);
            
            return tokenId;
        }

        public virtual ByteString[] CreateBatchTokens(UInt160[] receivers, string[] data, int[] amount)
        {
            if (receivers.Length != data.Length && receivers.Length != amount.Length)
            {
                throw new Exception("Recievers array size is not equal data array size or amount array size.");
            }
            
            List<ByteString> tokenIds = new();

            for (int i = 0; i < data.Length; i++)
            {
                tokenIds.Add(CreateToken(receivers[i], amount[i], data[i]));
            }
            
            return tokenIds;
        }

        [Safe]
        public static UInt160 GetOwner()
        {
            return (UInt160)ContractMap.Get(ownerKey);
        }

        public static bool TransferOwnership(UInt160 newOwner)
        {
            if (!newOwner.IsValid)
            {
                throw new Exception("The new owner address is invalid");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OwnershipTransferred(tx.Sender, newOwner);
            ContractMap.Put(ownerKey, newOwner);

            return true;
        }

        [DisplayName("block")]
        public static bool Block(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            BlocklistMap.Put(userAddress, trueInByteString);
            IsUserBlocked(userAddress, true);
            return true;
        }

        [DisplayName("unblock")]
        public static bool UnBlock(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            BlocklistMap.Delete(userAddress);
            IsUserBlocked(userAddress, false);

            return true;
        }

        [DisplayName("isblocked")]
        public static bool IsBlocked(UInt160 userAddress)
        {
            var blocked = BlocklistMap.Get(userAddress);

            return blocked is not null && blocked == trueInByteString;
        }

        [DisplayName("updateMarketplaceAddress")]
        public static bool UpdateMarketplaceAddress(UInt160 newMarketplaceAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            ContractMap.Put(marketplaceKey, newMarketplaceAddress);

            return true;
        }

        public static UInt160 GetMarketplaceAddress()
        {
            return (UInt160)ContractMap.Get(marketplaceKey);
        }


        [DisplayName("pause")]
        public static bool Pause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            ContractMap.Put(pausedKey, trueInByteString);
            IsContractPaused(true);

            return true;
        }

        [DisplayName("unpause")]
        public static bool Unpause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            ContractMap.Delete(pausedKey);
            IsContractPaused(false);

            return true;
        }

        [DisplayName("paused")]
        public static bool Paused()
        {
            var paused = ContractMap.Get(pausedKey);

            return paused == trueInByteString;
        }

        [DisplayName("_deploy")]
        public virtual void Deploy(object data, bool update)
        {
            if (update)
            {
                return;
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;
            ContractMap.Put(ownerKey, tx.Sender);
            ContractMap.Put(termsKey, "Terms and Conditions of Use \nNo intellectual property rights are being transferred to the NFT buyer. More specifically, the buyer obtains none of the following rights: \n - No Copyright \n - No Commercialization Rights \n - No Derivative Rights \n - No Rights to Claim Any Royalties upon Resale of the NFT \nThe NFT buyer has only the following rights: \n - Right to Have, Hold, Admire and Display the NFT in Private, Non-Commercial Settings \n - Right to Resell the NFT Acquired, Subject to These Same Terms and Conditions");

            if (data is not null) {
                Map<string, string> customData = (Map<string, string>)data;
                if (customData["to"] is not null &&
                    customData["image"] is not null &&
                    customData["name"] is not null &&
                    customData["description"] is not null &&
                    customData["amount"] is not null) {
                    var amount = (BigInteger)(ByteString)customData["amount"];
                    var to = (UInt160)(ByteString)customData["to"];
                    
                    Mint(to, NewTokenId(), amount, new TokenState
                    {
                        Creator = tx.Sender,
                        Name = customData["name"],
                        Image = customData["image"],
                        Description = customData["description"],
                        CreatedTime = Runtime.Time
                    });
                } else {
                    throw new InvalidOperationException("Insufficient NFT creation payload data");
                }
            }
        }

        public static bool Update(ByteString nefFile, string manifest, object data = null)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }

        protected override void Mint(UInt160 account, ByteString tokenId, BigInteger amount, TokenState token)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            tokenMap[tokenId] = StdLib.Serialize(token);
            PreTransfer(UInt160.Zero, account, tokenId, amount, null);
            UpdateBalance(account, tokenId, amount);
            UpdateTotalSupply(+1);
            PostTransfer(null, account, tokenId, amount, null);

            if(GetMarketplaceAddress() != UInt160.Zero && !IsApprovedForAll(account, GetMarketplaceAddress()))
            {
                InApprovalForAll(account, GetMarketplaceAddress(), true);
            }
        }

        protected override void PreTransfer(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            if (Paused())
            {
                throw new Exception("Contract has been paused");
            }

            if (IsBlocked(from) || IsBlocked(to))
            {
                throw new Exception("User 'from' or 'to' has been blocked");
            }
        }
    }

    public class TokenState : Dep1155TokenState
    {
        public string Description;
        public string Image;
        public ulong CreatedTime;
        public UInt160 Creator;
    }
}
