import { Injectable, Logger, HttpException, ConflictException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { UserInfoDto } from './dto/user-info.dto';
import { Repository } from 'typeorm';
import { UserInfoEntity } from './entities/user-info.entity';
import { OpenApiService } from '@/openapi/openapi.service';
import { TwitterInfoEntity } from '@/social-auth/entities/twitter-info.entity';
import { SocialInfoDto } from './dto/social-info.dto';

@Injectable()
export class UserInfoService {
  private readonly logger = new Logger(UserInfoService.name);

  constructor(
    @InjectRepository(UserInfoEntity)
    private readonly userInfoRepository: Repository<UserInfoEntity>,
    @InjectRepository(TwitterInfoEntity)
    private readonly twitterInfoRepository: Repository<TwitterInfoEntity>,
    private readonly openApiService: OpenApiService
  ) {}

  async create(address: string, userInfoDto: UserInfoDto): Promise<UserInfoEntity> {
    this.logger.debug(`Creating/Updating: ${address} -> ${JSON.stringify(userInfoDto)}`)
    const userInfo: UserInfoEntity = this.userInfoRepository.create({
      address,
      name: userInfoDto.name ?? '',
      surname: userInfoDto.surname ?? '',
      email: userInfoDto.email ?? '',
      avatar: userInfoDto.avatar ?? ''
    });

    const saved = await this.userInfoRepository.save(userInfo);
    return saved;
  }
  
  async createSocial(address: string, socialInfoDto: SocialInfoDto): Promise<UserInfoEntity> {
    this.logger.debug(`Creating/Updating: ${address} -> ${JSON.stringify(socialInfoDto)}`)
    // Get existent if exists
    const existent = await this.userInfoRepository.findOne(address);

    const userInfo: UserInfoEntity = this.userInfoRepository.create({
      address,
      dvitaId: socialInfoDto.dvitaId ?? '',
      twitter: socialInfoDto.twitter ?? ''
    });

    await this.updateCanonicalNames(userInfo, existent);

    const saved = await this.userInfoRepository.save(userInfo);
    return saved;
  }

  findOne(address: string): Promise<UserInfoEntity | undefined> {
    return this.userInfoRepository.findOne(address);
  }

  private async updateCanonicalNames(userInfo: UserInfoEntity, existent?: UserInfoEntity): Promise<void> {
    // TODO: Bulk update

    // 1. Email
    // TODO: Disabled for now: To add Mail confirmation flow first
    // if (userInfo.email) {
    //   try {
    //     const emailNormalized = userInfo.email.trim();
    //     await this.updateCanonicalNameImpl(emailNormalized, userInfo.address);
    //   } catch(e) {
    //     this.handleOpenApiError(e);
    //   }
    // }

    // 2. Dvita identity aka 'xyz.id.dvita.com'
    if (userInfo.dvitaId) {
      try {
        const dvitaIdNormalized = `${userInfo.dvitaId.trim()}.id.dvita.com`;
        await this.updateCanonicalNameImpl(dvitaIdNormalized, userInfo.address);
      } catch(e) {
        this.handleOpenApiError(e);
      }
    } else if (existent?.dvitaId && existent?.address) {
      try {
        const dvitaIdNormalized = `${existent.dvitaId.trim()}.id.dvita.com`;
        await this.unregisterCanonicalNameImpl(dvitaIdNormalized, existent.address);
      } catch(e) {
        this.handleOpenApiError(e, false);
      }
    }

    // 3. Social auth:> Twitter
    if (userInfo.twitter) {
      try {
        const twitterNormalized = `@${userInfo.twitter}`;
        await this.updateCanonicalNameImpl(twitterNormalized, userInfo.address);
      } catch(e) {
        this.handleOpenApiError(e);
      }
    } else if (existent?.twitter && existent?.address) {
      try {
        const twitterNormalized = `@${existent.twitter}`;
        await this.unregisterCanonicalNameImpl(twitterNormalized, existent.address);
        await this.twitterInfoRepository.delete(existent.address);
      } catch(e) {
        this.handleOpenApiError(e, false);
      }
    }
  }

  private async updateCanonicalNameImpl(cname: string, address: string): Promise<void> {
    const canBeResolved = await this.checkIfResolved(cname, address);
    if (!canBeResolved) {
      this.logger.debug(`Updating: ${cname} -> ${address}`);
      const result = await this.openApiService.registerCanonicalName(cname, address);
      this.logger.debug(`Updated: ${cname} -> ${address}: ${JSON.stringify(result)}`);
    } else {
      this.logger.debug(`NOT Updated: ${cname} -> ${address}: ${canBeResolved}`);
    }
  }

  private async unregisterCanonicalNameImpl(cname: string, address: string): Promise<void> {
    const canBeResolved = await this.checkIfResolved(cname, address);
    if (canBeResolved) {
      this.logger.debug(`Unregistering: ${cname} -> ${address}`);
      const result = await this.openApiService.unregisterCanonicalName(cname);
      this.logger.debug(`Unregistered: ${cname} -> ${address}: ${JSON.stringify(result)}`);
    } else {
      this.logger.debug(`NOT Unregistered: ${cname} -> ${address}: ${canBeResolved}`);
    }
  }

  private async checkIfResolved(cname: string, toAddress?: string): Promise<boolean> {
    try {
      var result = await this.openApiService.resolveCanonicalName(cname);
      
      // Regular resolution
      if (!toAddress) {
        return result.cname === cname && result.address !== '<unknown address>';
      }
    } catch(e) {
      this.logger.error(e);
      return false;
    }

    // Hoisted out of try/catch
    if (!result) {
      return false;
    }

    if (result.address === '<unknown address>') {
      return false;
    }
    
    // 'To exact address' resolution
    if (result.cname === cname && result.address === toAddress!) {
      return true;
    } else {
      this.logger.error(`Error: Name already exists: ${JSON.stringify(result)} for ${cname} : ${toAddress}.`);
      throw new ConflictException(`Error: Name already exists.`);
    }
  }

  //
  // Fix for: Nest-to-Nest HTTP Ecxeption misaligned.
  //
  private handleOpenApiError(error: any, rethrow: boolean = true): void {
      this.logger.error(error);

      if (!rethrow) {
        return;
      }

      // Re-throw as Generic Exception
      if (!error?.isAxiosError) {
        throw error;
      }

      const code = error?.response?.data?.statusCode ?? 500;
      const message = error?.response?.data?.message ?? 'Internal Server Error';

      console.log(`Throwing: ${code} : ${message}`);

      // Re-throw as HTTP Exception
      throw new HttpException(message, code);
  }
}
