import { Request } from 'express';
import { Body, Controller, Get, Post, Param, Req, UseGuards } from '@nestjs/common';
import {
  ApiBearerAuth,
  ApiOperation,
  ApiResponse,
  ApiTags
} from '@nestjs/swagger';
import { AuthGuard, getAuthTokenSigner } from '@/auth/auth.guard';
import { UserInfoDto } from './dto/user-info.dto';
import { UserInfoEntity } from './entities/user-info.entity';
import { UserInfoService } from './user-info.service';
import { SocialInfoDto } from './dto/social-info.dto';

@ApiBearerAuth()
@ApiTags('user-info')
@Controller('user-info')
export class UserInfoController {
  constructor(private readonly userInfoService: UserInfoService) {}

  @Post()
  @UseGuards(AuthGuard)
  @ApiOperation({ summary: 'Create or update user info' })
  @ApiResponse({ status: 403, description: 'Forbidden' })
  @ApiResponse({ status: 404, description: 'Not Found' })
  @ApiResponse({ status: 409, description: 'Conflict' })
  async create(@Body() userInfoDto: UserInfoDto, @Req() req: Request): Promise<UserInfoEntity> {
    const address = getAuthTokenSigner(req);
    const created = await this.userInfoService.create(address, userInfoDto);
    return created;
  }

  @Post('social')
  @UseGuards(AuthGuard)
  @ApiOperation({ summary: 'Create or update user social account info' })
  @ApiResponse({ status: 403, description: 'Forbidden' })
  @ApiResponse({ status: 404, description: 'Not Found' })
  @ApiResponse({ status: 409, description: 'Conflict' })
  async createSocial(@Body() socialInfoDto: SocialInfoDto, @Req() req: Request): Promise<UserInfoEntity> {
    const address = getAuthTokenSigner(req);
    const created = await this.userInfoService.createSocial(address, socialInfoDto);
    return created;
  }

  @Get(':address')
  @ApiResponse({ status: 200, description: 'User info', type: UserInfoEntity })
  @ApiResponse({ status: 403, description: 'Forbidden' })
  async findOne(@Param('address') address: string, @Req() req: Request): Promise<UserInfoEntity> {
    let tokenSignerAddress = '';
    try {
        tokenSignerAddress = getAuthTokenSigner(req);
    } catch (err) {
      console.log('Not allowed to view private user data. Will return public data. Auth error was:', err);
    }
    const userInfo: UserInfoEntity | undefined = await this.userInfoService.findOne(address);
    const isAllowedToViewPrivateInfo = address === tokenSignerAddress;
    return {
      name: '',
      surname: '',
      email: '',
      dvitaId: '',
      twitter:'',
      avatar: '',
      createdDate: new Date(),
      modifiedDate: new Date(),

      // public info
      address: address,

      // private info
      ...(isAllowedToViewPrivateInfo ? userInfo : {}),
    };
  }
}
