import { OpenApiService } from '@/openapi/openapi.service';
import { SocialRewardsService } from '../social-rewards/social.rewards.service';
import { UserInfoEntity } from '@/user-info/entities/user-info.entity';
import { BadRequestException, Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { HttpAdapterHost } from '@nestjs/core';
import { InjectRepository } from '@nestjs/typeorm';
import { OnGatewayConnection, OnGatewayDisconnect, OnGatewayInit, WebSocketGateway, WebSocketServer } from '@nestjs/websockets';
import { Request, Response } from 'express';
import { IOAuth2RequestTokenResult, TwitterApi } from 'twitter-api-v2';
import { Repository } from 'typeorm';
import { Server, WebSocket } from 'ws';
import { TwitterInfoEntity } from './entities/twitter-info.entity';

type IOAuth2RequestTokenResultEx = IOAuth2RequestTokenResult & { blockchainAddress: string };

/*
AuthTwitterService
See:
https://developer.twitter.com/en/docs/authentication/guides/log-in-with-twitter
https://developer.twitter.com/en/docs/authentication/guides/authentication-best-practices
https://developer.twitter.com/en/docs/authentication/oauth-1-0a/obtaining-user-access-tokens
https://github.com/PLhery/node-twitter-api-v2/blob/master/doc/auth.md
*/
@Injectable()
// See:
//    https://docs.nestjs.com/websockets/gateways
//    https://docs.nestjs.com/websockets/adapter
//    https://github.com/nestjs/nest/blob/master/sample/16-gateways-ws/src/events/events.gateway.ts  
@WebSocketGateway(
    process.env.NODE_ENV === 'production' ?
        undefined :
        5001,
    { transports: ['websocket'], cors: { origin: '*' } }
)
export class AuthTwitterService implements OnModuleInit, OnGatewayInit, OnGatewayConnection, OnGatewayDisconnect {
    // TODO:
    //  1. Remove entry on success
    //  2. Remove entry on failure
    //  3. Remove entry by timeout
    private sessions: Record<string, IOAuth2RequestTokenResultEx> = {};

    @WebSocketServer()
    private readonly wsServer: Server;

    private readonly logger = new Logger(AuthTwitterService.name);
    private readonly redirectUri = process.env.TWITTER_REDIRECT_URI as string;
    private readonly twitterApiOptions = { 
        clientId: process.env.TWITTER_CLIENTID as string, 
        clientSecret: process.env.TWITTER_CLIENSECRET as string
    };

    private client?: TwitterApi;

    constructor(
        // See: https://docs.nestjs.com/faq/http-adapter
        private readonly adapterHost: HttpAdapterHost,
        @InjectRepository(TwitterInfoEntity)
        private readonly userInfoRepository: Repository<UserInfoEntity>,
        @InjectRepository(TwitterInfoEntity)
        private readonly twitterInfoRepository: Repository<TwitterInfoEntity>,
        private readonly openApiService: OpenApiService,
        private readonly socialRewardsService: SocialRewardsService,
    ) {
    }

    onModuleInit(): any {
        this.client = new TwitterApi(this.twitterApiOptions);
    }

    onModuleDestroy(): any {
        this.client = undefined;
    }

    afterInit(server: Server): any {
        this.logger.log(`WS Gateway initialized: ${JSON.stringify(server.options)}`);
    }

    handleConnection(client: WebSocket, ...args: any[]): any {
        this.logger.log(`WS Client connected: ${client.readyState}`);

        this.logger.log(`WS Sending to ${this.wsServer.clients.size}`);
        this.wsServer.clients.forEach((c: WebSocket) => c.send(JSON.stringify({ping: true})));

        client.send(JSON.stringify({ping: true}));
    }
      
    handleDisconnect(client: WebSocket): any {
        this.logger.log(`WS Client disconnected: ${client.readyState}`);
    }

    async createAuthUrl(req: Request, address: string): Promise<string> {
        let authLink = await this.client!.generateOAuth2AuthLink(
            this.redirectUri, 
            { scope: ['tweet.read', 'users.read', 'offline.access'] }
        ) as IOAuth2RequestTokenResultEx;
        authLink.blockchainAddress = address;
        this.logger.debug(JSON.stringify(authLink));
        this.sessions[authLink.state] = authLink;
        return authLink.url;
    }

    async twitterAuthCallback(req: Request, res: Response): Promise<void> {
        // Request level
        const { state, code } = req.query as {
            state: string;
            code: string;
        };

        if (!state || !code) {
            this.logger.warn('You denied the app or your received session expired!');
            throw new BadRequestException('You denied the app or your session expired!');
        }
        
        // DB/Persist/Session/Cookies/Whatewer level (see this.sessions).
        const storedSession = this.sessions[state];

        if (!storedSession || !storedSession.codeVerifier) {
            this.logger.warn('You denied the app or your stored session expired!');
            throw new BadRequestException('You denied the app or your session expired!');
        }

        if (state !== storedSession.state) {
            this.logger.warn('Stored tokens didnt match!');
            throw new BadRequestException('Stored tokens didnt match!');
        }

        // Obtain access token
        const loginResult = await this.client!.loginWithOAuth2({ code, codeVerifier: storedSession.codeVerifier, redirectUri: this.redirectUri });
        const accessToken = loginResult.accessToken;
        const refreshToken = loginResult.refreshToken;
        const expiresIn = loginResult.expiresIn;

        // Obtain 'me'
        const me = await loginResult.client.v2.me();
        const twitterUserId = me.data.id;
        const twitterUserName = me.data.username;

        // Persist User info  aka 'me' & its API access tokens
        if (twitterUserId && twitterUserName) {
            const twitterInfo = this.twitterInfoRepository.create({
                address: storedSession.blockchainAddress,
                twitterUserId,
                twitterUserName,
                accessToken,
                refreshToken,
                expiresIn
            });
            const savedTwitterInfo = await this.twitterInfoRepository.save(twitterInfo);

            // Update User Info entity too
            const userInfoFound =
                await this.userInfoRepository.findOne(storedSession.blockchainAddress) ??
                this.userInfoRepository.create({
                    address: storedSession.blockchainAddress,
                    twitter: twitterUserName
                });
            const savedUserInfo = await this.userInfoRepository.save(userInfoFound);

            this.logger.log('Saved:', JSON.stringify(savedTwitterInfo), JSON.stringify(savedUserInfo));
        } else {
            this.logger.warn('Wrong User object received!');
            throw new BadRequestException('Wrong User object received!');
        }

        // Start awarding user flow in async way
        this.awardUserForTwitterConnected(storedSession.blockchainAddress, twitterUserId, twitterUserName, accessToken)
            .then((res) => {
                this.logger.debug(`User awarded: ${storedSession.blockchainAddress} : ${twitterUserName} : ${JSON.stringify(res)}`);
            })
            .catch((e) => {
                this.logger.error(e, e.stack);
            });

        // Notify web via WS
        let response = { loginStatus: "Ok" };
        this.logger.log(`WS Sending to ${this.wsServer.clients.size}`);
        this.wsServer.clients.forEach((c: WebSocket) => c.send(JSON.stringify(response)));
        
        this.logger.log(response);

        // Respond to the twitter calback using injected 'response'
        res.redirect(`${process.env['PORTAL_BASE_URL']}/social/welcome/twitter`);
        res.end();
    };

    async getTwitterUsername(address: string): Promise<string | undefined> {
        this.logger.log('Twitter username for: ', address);
        const twitterInfo = await this.twitterInfoRepository.findOne(address);
        return twitterInfo ? twitterInfo.twitterUserName : '';
    }

    private async awardUserForTwitterConnected(
        address: string,
        userId: string,
        userSocialHandle: string,
        accessToken: string
    ): Promise<any> {
        try {
            return await this.socialRewardsService.awardUser('twitter', address, userId, userSocialHandle, accessToken);
        } catch(e: any) {
            this.logger.error(e, e.stack);
        }
    }
}
