import { HttpService } from "@nestjs/axios";
import { Injectable, Logger } from "@nestjs/common";
import { lastValueFrom } from "rxjs";

export interface RegisterCanonicalNameRequestDto {
    cname: string;
    address: string;
}

export interface RegisterCanonicalNameResultDto {
    cname: string;
    address: string;
    txHash: string;
}

export interface UnregisterCanonicalNameRequestDto {
    cname: string;
}

export interface UnregisterCanonicalNameResultDto {
    cname: string;
    txHash: string;
}

export interface ResolveCanonicalNameResultDto {
    cname: string;
    address: string;
}

@Injectable()
export class OpenApiService {
    private readonly logger = new Logger(OpenApiService.name);

    private static readonly openapiEndpoint = process.env.OPENAPI_ENDPOINT!;

    constructor(
        private readonly httpService: HttpService
    ) {
        this.logger.debug('Using:', OpenApiService.openapiEndpoint);
    }

    async registerCanonicalName(cname: string, address: string): Promise<RegisterCanonicalNameResultDto> {
        const requestData: RegisterCanonicalNameRequestDto = {
            cname,
            address,
        };
        const response = await lastValueFrom(
            this.httpService.post<RegisterCanonicalNameResultDto>(`${OpenApiService.openapiEndpoint}/cnr/register`, requestData)
        );
        return response.data;
    }

    async unregisterCanonicalName(cname: string): Promise<UnregisterCanonicalNameResultDto> {
        const requestData: UnregisterCanonicalNameRequestDto = {
            cname,
        };
        const response = await lastValueFrom(
            this.httpService.post<UnregisterCanonicalNameResultDto>(`${OpenApiService.openapiEndpoint}/cnr/unregister`, requestData)
        );
        return response.data;
    }

    async resolveCanonicalName(cname: string): Promise<ResolveCanonicalNameResultDto> {
        const response = await lastValueFrom(
            this.httpService.get<ResolveCanonicalNameResultDto>(`${OpenApiService.openapiEndpoint}/cnr/resolve/${cname}`)
        );
        return response.data;
    }
}