// Copyright (C) 2015-2021 The Neo Project.
// 
// The neo is free software distributed under the MIT software license, 
// see the accompanying file LICENSE in the main directory of the
// project or http://www.opensource.org/licenses/mit-license.php 
// for more details.
// 
// Redistribution and use in source and binary forms with or without
// modifications are permitted.

using Neo.IO;
using System;
using System.IO;

namespace Neo.SmartContract
{
    /// <summary>
    /// Used to build storage keys for native contracts.
    /// </summary>
    public class KeyBuilder
    {
        private readonly int id;
        private readonly MemoryStream stream = new();

        /// <summary>
        /// Initializes a new instance of the <see cref="KeyBuilder"/> class.
        /// </summary>
        /// <param name="id">The id of the contract.</param>
        /// <param name="prefix">The prefix of the key.</param>
        public KeyBuilder(int id, byte prefix)
        {
            this.id = id;
            this.stream.WriteByte(prefix);
        }

        /// <summary>
        /// Adds part of the key to the builder.
        /// </summary>
        /// <param name="key">Part of the key.</param>
        /// <returns>A reference to this instance after the add operation has completed.</returns>
        public KeyBuilder Add(ReadOnlySpan<byte> key)
        {
            stream.Write(key);
            return this;
        }

        /// <summary>
        /// Adds part of the key to the builder.
        /// </summary>
        /// <param name="key">Part of the key.</param>
        /// <returns>A reference to this instance after the add operation has completed.</returns>
        public KeyBuilder Add(ISerializable key)
        {
            using (BinaryWriter writer = new(stream, Utility.StrictUTF8, true))
            {
                key.Serialize(writer);
                writer.Flush();
            }
            return this;
        }

        /// <summary>
        /// Adds part of the key to the builder.
        /// </summary>
        /// <typeparam name="T">The type of the <paramref name="key"/> parameter.</typeparam>
        /// <param name="key">Part of the key.</param>
        /// <returns>A reference to this instance after the add operation has completed.</returns>
        unsafe public KeyBuilder Add<T>(T key) where T : unmanaged
        {
            return Add(new ReadOnlySpan<byte>(&key, sizeof(T)));
        }

        /// <summary>
        /// Adds part of the key to the builder with big-endian.
        /// </summary>
        /// <typeparam name="T">The type of the <paramref name="key"/> parameter.</typeparam>
        /// <param name="key">Part of the key.</param>
        /// <returns>A reference to this instance after the add operation has completed.</returns>
        unsafe public KeyBuilder AddBigEndian<T>(T key) where T : unmanaged
        {
            ReadOnlySpan<byte> buffer = new(&key, sizeof(T));
            for (int i = buffer.Length - 1; i >= 0; i--)
                stream.WriteByte(buffer[i]);
            return this;
        }

        /// <summary>
        /// Gets the storage key generated by the builder.
        /// </summary>
        /// <returns>The storage key.</returns>
        public byte[] ToArray()
        {
            using (stream)
            {
                return StorageKey.CreateSearchPrefix(id, stream.ToArray());
            }
        }

        public static implicit operator StorageKey(KeyBuilder builder)
        {
            using (builder.stream)
            {
                return new StorageKey
                {
                    Id = builder.id,
                    Key = builder.stream.ToArray()
                };
            }
        }
    }
}
